using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Net;
using System.Runtime.InteropServices;
using System.Threading;
using Microsoft.WindowsAzure;
using Microsoft.WindowsAzure.Diagnostics;
using Microsoft.WindowsAzure.ServiceRuntime;
using Microsoft.WindowsAzure.Storage;

namespace WorkerRole
{
    public class WorkerRole : RoleEntryPoint
    {

        readonly CancellationTokenSource _cancelSource = new CancellationTokenSource();
        readonly FileSystemWatcher _fileSystemWatcher = new FileSystemWatcher();
        private readonly string _pathToMonitor;
        private readonly string _username;
        private readonly string _password;
        private readonly string _domainName;

        public WorkerRole()
        {
            _pathToMonitor = CloudConfigurationManager.GetSetting("PathToMonitor");
            _username = CloudConfigurationManager.GetSetting("Username");
            _password = CloudConfigurationManager.GetSetting("Password");
            _domainName = String.IsNullOrWhiteSpace(CloudConfigurationManager.GetSetting("DomainName")) ? Environment.MachineName : CloudConfigurationManager.GetSetting("DomainName");
        }

        public override void Run()
        {
            _cancelSource.Token.WaitHandle.WaitOne();
        }

        void FileSystemWatcher_OnFileRenamed(object sender, RenamedEventArgs e)
        {
            Trace.TraceInformation(string.Format("File {0} renamed", e.FullPath));
        }

        void FileSystemWatcher_OnFileCreated(object sender, FileSystemEventArgs e)
        {
            Trace.TraceInformation(string.Format("File {0} created", e.FullPath));
        }

        private void InitializeFileWatcher()
        {
            _fileSystemWatcher.Path = _pathToMonitor;
            _fileSystemWatcher.Created += FileSystemWatcher_OnFileCreated;
            _fileSystemWatcher.Renamed += FileSystemWatcher_OnFileRenamed;
            _fileSystemWatcher.EnableRaisingEvents = true;
        }


        public override bool OnStart()
        {
            new Action(InitializeFileWatcher).ExecuteAs(_username, _password, _domainName);
            return base.OnStart();
        }

        public override void OnStop()
        {
            _fileSystemWatcher.EnableRaisingEvents = false;
            _fileSystemWatcher.Dispose();
            _cancelSource.Cancel();
        }
    }
}
