﻿using System;
using System.Diagnostics;

namespace PerformanceCounters
{
    [Serializable]
    public class PerformanceCounterManagement : IPerformanceCounterManagement
    {
        public virtual void CreatePerformanceCounterCategory(string categoryName, string counterName, string categoryHelp = null, string performanceCounterHelperText = null)
        {
            if (!PerformanceCounterCategory.Exists(categoryName))
            {
                Trace.TraceInformation(@"Creating performance counter category '{0}' and performance counter '\{0}(*)\{1}'", categoryName, counterName);
                PerformanceCounterCategory.Create(categoryName, categoryHelp, PerformanceCounterCategoryType.Unknown, counterName, performanceCounterHelperText);
                Trace.TraceInformation(@"Performance counter category '{0}' and performance counter '\{0}(*)\{1}' created", categoryName, counterName);
            }

            Trace.TraceWarning(@"Performance counter category '{0}' already exists! Performance counter '\{0}(*)\{1}' will not be created!", categoryName, counterName);
        }

        public virtual void DeleteCategory(string categoryName)
        {
            if (PerformanceCounterCategory.Exists(categoryName))
            {
                Trace.TraceInformation("Deleting performance category '{0}'", categoryName);
                PerformanceCounterCategory.Delete(categoryName);
                Trace.TraceInformation("Category '{0}' deleted", categoryName);
            }
        }

        public virtual void DeleteNotUnusedPerformanceCounterInstances(string categoryName, string counterName, params string[] instanceNames)
        {
            foreach (var instanceName in instanceNames)
            {
                Trace.TraceInformation(@"Deleting performance counter instance \{0}({1})\{2}", categoryName, instanceName, counterName);

                if (PerformanceCounterCategory.InstanceExists(instanceName, categoryName))
                {
                    var performanceCounter = CreateOrGetPerformanceCounterInstance(categoryName, counterName, instanceName);
                    performanceCounter.RemoveInstance();
                    Trace.TraceInformation(@"Performance counter instance \{0}({1})\{2} deleted", categoryName, instanceName, counterName);
                }
                else
                {
                    Trace.TraceInformation(@"Performance counter instance \{0}({1})\{2} can't be deleted because don't exists!", categoryName, instanceName, counterName);
                }
            }
        }

        public virtual PerformanceCounter CreateOrGetPerformanceCounterInstance(string categoryName, string counterName, string instanceName)
        {
            if (PerformanceCounterCategory.InstanceExists(instanceName, categoryName))
            {
                Trace.TraceWarning(@"Performance counter instance \{0}({1})\{2} don't exist! New instance will be created!", categoryName, instanceName, counterName);
            }

            return new PerformanceCounter(categoryName, counterName, instanceName, false);
        }
    }
}